/*
 **********************************************************************
 *  DLMS/COSEM components
 *
 *  i-cube software, 1966 Ayent, Switzerland
 *  infos@icube.ch
 *
 * (c) Copyright 2004-2016
 **********************************************************************
 */
package icube.demo;

import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;

import icube.ezhdlc.HDLCProtocolState;
import icube.ezhdlc.ICallbacks;
import icube.wrapper.WRAPPERException;
import icube.wrapper.WRAPPERParameters;
import icube.wrapper.WRAPPERProtocol;
import icube.xmlpdu.XmlPduInterface;
import icube.xmlpdu.infra.XmlPduException;

/**
 * Console application to demonstrate how to use the libraries.
 */
public class WrapperDemo implements ICallbacks, icube.wrapper.ICallbacks {

	private icube.wrapper.WRAPPERParameters wrapperParameters;
	private icube.wrapper.WRAPPERProtocol wrapper;
    private Socket socket;
	
	private long startTime = System.currentTimeMillis();
	private String Host;
	private int Port;

	public class DemoException extends Exception {
		public DemoException(String message) {
			super(message);
		}
	}

	/**
	 * Initialize parameters.
	 * <p>You should be <bold>very careful</bold> with the 'Client*' & 'Server*'
	 * parameters; refer to your meter documentation.
	 */
	public void initializeParameters()
    {
        this.wrapperParameters = new WRAPPERParameters();
		this.wrapperParameters.Callbacks = this;
		this.wrapperParameters.ClientAddress = 0x10;
		this.wrapperParameters.ServerAddress = 1;

		this.Host = "127.0.0.1";	// FIXME update according to your configuration
		this.Port = 4058;
    }
	
	/**
	 * Processes an XML file.
	 * 
	 * @param filePath the file path
	 * @throws WRAPPERException 
	 */
	public boolean processXmlFile(String filePath) throws XmlPduException, 
									IOException, WRAPPERException {
		boolean success = false;
	    StringBuffer outXml = new StringBuffer();
	    StringBuffer inXml;
	    byte[] outPdu, inPdu;
	
	    System.out.println("--------        --------        --------        --------        --------");
	    System.out.printf("Processing: %s\n", filePath);
	
	//    try
	//    {
			FileReader F = new FileReader(filePath);
			try {
				while (F.ready()) {
					outXml.append((char) F.read());
				}
			} finally {
				F.close();
			}
			
	        System.out.printf(outXml.toString());
			outPdu = XmlPduInterface.XmlToPdu(outXml);
	        System.out.println();
	
	        // Send to meter and get response
			inPdu = this.wrapper.sendReceive(outPdu);
	        inXml = XmlPduInterface.PduToXml(inPdu);
	
	        System.out.printf(inXml.toString());
	        success = true;
	//    }
	//    catch (Exception e)
	//    {
	//    	System.out.print(e.getMessage());
	//    }
	//    System.out.println();
	    return success;
	}

	private void doIt(String[] args) throws DemoException {
	    InputStream inputStream;
	    OutputStream outputStream;

    	System.out.println(icube.ezhdlc.Version.getVersionStr());
    	System.out.println(icube.xmlpdu.Version.getVersionStr());
    	System.out.println(icube.wrapper.Version.getVersionStr());

    	this.initializeParameters();
		
		// open socket
        try
        {
        	this.socket = new Socket();
        	this.socket.bind(null);
        	this.socket.connect(new InetSocketAddress(
        						this.Host, 
        						this.Port), 
        						3000);
        	this.socket.setSoTimeout(3000);

        	inputStream = this.socket.getInputStream();
			outputStream = this.socket.getOutputStream();
        }
        catch (Exception e)
        {
			throw new DemoException("TCP/IP - error " + e.getMessage());
        }
		
		try {
        	this.wrapper = new WRAPPERProtocol(this.wrapperParameters, inputStream, outputStream);

            for (int i = 0; i < args.length; i++) {
            	boolean success = this.processXmlFile(args[i]);
            	if (!success)
            		break;
            }
			
			
			System.out.println("Disconnecting");
			
		} catch (XmlPduException e) {
			throw new DemoException("XmlPduException - " + e.getMessage());
		} catch (WRAPPERException e) {
			throw new DemoException("Wrapper - IOException - " + e.getMessage());
		} catch (IOException e) {
			throw new DemoException("IOException - " + e.getMessage());
		} finally {
			if (this.socket != null)
		        try
		        {
		            this.socket.close();
		            this.socket = null;
		            
		        }
		        catch (Exception e)
		        {
					throw new DemoException("TCP/IP - close - " + e.getMessage());
	        }
		}
	}

	/**
	 * Launches the demo
	 */
	public static void main(String[] args) {

		try {
			WrapperDemo demo = new WrapperDemo();
			demo.doIt(args);
		} catch (DemoException e) {
			e.printStackTrace();
		}
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onReceivedStream(java.lang.Object)
	 */
	@Override
	public void onReceivedStream(byte[] stream) {
	    this.logLine("CB_R: " + icube.ezhdlc.Utils.bytesToHexString((byte[])stream));
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onSentStream(java.lang.Object)
	 */
	@Override
	public void onSentStream(byte[] stream) {
	    this.logLine("CB_S: " + icube.ezhdlc.Utils.bytesToHexString((byte[])stream));
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onStateChange(icube.ezhdlc.HDLCProtocolState)
	 */
	@Override
	public void onStateChange(HDLCProtocolState state) {
		this.logLine("CB_s: " + state.toString());
	}

	private void logLine(String log) {
		long deltaTime = System.currentTimeMillis() - this.startTime;
		String dtStr = String.format("%03d", deltaTime);
		if (dtStr.length() < 4) {
			dtStr = "0." + dtStr;
		} else {
			int dotPos = dtStr.length() -3;
			dtStr = dtStr.substring(0, dotPos) + "." + dtStr.substring(dotPos);
		
		}
		System.out.printf("%9s, %s\n", dtStr, log);
	}

	
}
