/*
 **********************************************************************
 *  DLMS/COSEM components
 *
 *  i-cube software, 1966 Ayent, Switzerland
 *  infos@icube.ch
 *
 * (c) Copyright 2004-2016
 **********************************************************************
 */
package icube.demo;

import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.fazecast.jSerialComm.*;
import icube.ezhdlc.EzHdlcException;
import icube.ezhdlc.HDLCParameters;
import icube.ezhdlc.HDLCProtocol;
import icube.ezhdlc.HDLCProtocolState;
import icube.ezhdlc.ICallbacks;
import icube.xmlpdu.XmlPduInterface;
import icube.xmlpdu.infra.XmlPduException;


/**
 * This demo shows how to communicate with a Meter using the Hdlc communication
 * profile and a serial-port.
 * 
 * The meter's HDLC parameters are defined in initializeParameters().
 * The serial port parameters are defined in doIt().
 * 
 * It processes all xml files given as arguments as follow:
 * - convert to pdu
 * - send pdu and receive it's response
 * - convert pdu response to xml.
 */
public class HdlcDemo implements ICallbacks {

	private HDLCParameters hdlcParameters;
	private HDLCProtocol hdlc;
	private com.fazecast.jSerialComm.SerialPort serialPort;
	private long startTime = System.currentTimeMillis();
	private String portName;

	public class DemoException extends Exception {
		public DemoException(String message) {
			super(message);
		}
	}

	/**
	 * Initialize parameters.
	 * <p>Adapt to your meter parameters.</bold>
	 */
	public void initializeParameters()
    {
        this.hdlcParameters = new HDLCParameters();
        this.hdlcParameters.Callbacks = this;
	
        this.hdlcParameters.ClientMACAddress = 0x10;		// Public client
        this.hdlcParameters.ServerAddressSize = 1;			// Assume a L+G meter
        this.hdlcParameters.ServerUpperMACAddressValue = 1;	// Management logical device
        this.hdlcParameters.ServerLowerMACAddressValue = 0;	// Not used when address size = 1
	    	
	    this.portName = "COM15";
	}
	
	/**
	 * Processes an XML file.
	 * 
	 * @param filePath the file path
	 */
	public boolean processXmlFile(String filePath) throws XmlPduException, 
									EzHdlcException, IOException {
		boolean success = false;
	    StringBuffer outXml = new StringBuffer();
	    StringBuffer inXml;
	    byte[] outPdu, inPdu;
	
	    System.out.println("--------        --------        --------        --------        --------");
	    System.out.printf("Processing: %s\n", filePath);
	
	//    try
	//    {
			FileReader F = new FileReader(filePath);
			try {
				while (F.ready()) {
					outXml.append((char) F.read());
				}
			} finally {
				F.close();
			}
			
	        System.out.printf(outXml.toString());
			outPdu = XmlPduInterface.XmlToPdu(outXml);
	        System.out.println();
	
	        // Send to meter and get response
			inPdu = this.hdlc.sendReceive(outPdu);
	        inXml = XmlPduInterface.PduToXml(inPdu);
	
	        System.out.printf(inXml.toString());
	        success = true;
	//    }
	//    catch (Exception e)
	//    {
	//    	System.out.print(e.getMessage());
	//    }
	//    System.out.println();
	    return success;
	}

	/**
	 * Do the test.
	 * 
	 * @param args
	 * @throws DemoException
	 */
	private void doIt(String[] args) throws DemoException {
	    InputStream inputStream;
	    OutputStream outputStream;

		this.initializeParameters();
		
    	System.out.println(icube.ezhdlc.Version.getVersionStr());
    	System.out.println(icube.xmlpdu.Version.getVersionStr());
    	System.out.println(icube.wrapper.Version.getVersionStr());

		try {
			serialPort = SerialPort.getCommPort(portName);
			if (serialPort.isOpen())
				throw new DemoException("COMChannel() - PortInUse");
			// serialPort.setComPortParameters(9600,8,2,SerialPort.NO_PARITY);
			serialPort.setComPortParameters(9600, 8, 1, SerialPort.NO_PARITY);
			serialPort.setComPortTimeouts(SerialPort.TIMEOUT_READ_BLOCKING,  3000,  1000);
			serialPort.openPort();
			inputStream = this.serialPort.getInputStream();
			outputStream = this.serialPort.getOutputStream();
			
		} catch (SerialPortInvalidPortException e) {
			throw new DemoException("SerialPort - InvalidPortException" + e.getMessage());
		}
		
		try {
			this.hdlc = new HDLCProtocol(this.hdlcParameters, inputStream, outputStream);
			this.hdlc.connect();
			System.out.println("Connected to: " + this.portName);

            for (int i = 0; i < args.length; i++) {
            	boolean success = this.processXmlFile(args[i]);
            	if (!success)
            		break;
            }
			
			
			System.out.println("Disconnecting");
			this.hdlc.disconnect();
			
		} catch (EzHdlcException e) {
			throw new DemoException("EzHdlcException - " + e.getMessage());
		} catch (IOException e) {
			throw new DemoException("IOException - " + e.getMessage() + serialPort.getLastErrorCode());
		} catch (XmlPduException e) {
			throw new DemoException("XmlPduException - " + e.getMessage());
		} finally {
			if (this.serialPort != null)
				this.serialPort.closePort();
		}
	}

	/**
	 * Launches the demo
	 */
	public static void main(String[] args) {

		try {
			HdlcDemo demo = new HdlcDemo();
			demo.doIt(args);
		} catch (Exception e) {
			System.out.printf("%s\n", e.getMessage());
		}
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onReceivedStream(java.lang.Object)
	 */
	@Override
	public void onReceivedStream(byte[] stream) {
	    this.logLine("CB_R: " + icube.ezhdlc.Utils.bytesToHexString((byte[])stream));
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onSentStream(java.lang.Object)
	 */
	@Override
	public void onSentStream(byte[] stream) {
	    this.logLine("CB_S: " + icube.ezhdlc.Utils.bytesToHexString((byte[])stream));
	}

	/* (non-Javadoc)
	 * @see icube.ezhdlc.ICallbacks#onStateChange(icube.ezhdlc.HDLCProtocolState)
	 */
	@Override
	public void onStateChange(HDLCProtocolState state) {
		this.logLine("CB_s: " + state.toString());
	}

	private void logLine(String log) {
		long deltaTime = System.currentTimeMillis() - this.startTime;
		String dtStr = String.format("%03d", deltaTime);
		if (dtStr.length() < 4) {
			dtStr = "0." + dtStr;
		} else {
			int dotPos = dtStr.length() -3;
			dtStr = dtStr.substring(0, dotPos) + "." + dtStr.substring(dotPos);
		
		}
		System.out.printf("%9s, %s\n", dtStr, log);
	}

	
}
